/*!
 * unity-webgl.js v3.5.5
 * (c) 2024 Mervin<mengqing723@gmail.com>
 * Released under the MIT License.
 */
'use strict';

const isBrowser = typeof window !== 'undefined' && typeof window.document !== 'undefined';
const isPlainObject = arg => Object.prototype.toString.call(arg) === '[object Object]';
const msgPrefix = '[UnityWebgl]: ';
const log = (msg) => {
    console.log(msgPrefix, msg);
};
log.warn = (msg) => console.warn(msgPrefix, msg);
log.error = (msg) => console.error(msgPrefix, msg);
/**
 * query CanvasElement
 * @param {string | HTMLCanvasElement} canvas
 * @returns
 */
function queryCanvas(canvas) {
    if (canvas instanceof HTMLCanvasElement) {
        return canvas;
    }
    else if (typeof canvas === 'string') {
        return document.querySelector(canvas);
    }
    else {
        log.warn('CanvasElement not found.');
        return null;
    }
}

class EventBus {
    constructor(_eventMap = {}) {
        this._eventMap = _eventMap;
    }
    on(name, handler) {
        if (typeof handler !== 'function') {
            throw new TypeError(msgPrefix + 'The argument handler must be function');
        }
        this._eventMap[name] = handler;
        return this;
    }
    off(name) {
        delete this._eventMap[name];
        return this;
    }
    once(name, handler) {
        const ctx = this;
        function fn(...args) {
            if (typeof handler === 'function') {
                ctx.off(name);
                handler.apply(ctx, args);
            }
        }
        return this.on(name, fn);
    }
    emit(name, ...args) {
        const handler = this._eventMap[name];
        if (handler) {
            handler.apply(this, args);
        }
    }
    clear() {
        for (const name in this._eventMap) {
            delete this._eventMap[name];
        }
    }
}

/**
 * The unity loader
 * @param {string} src loaderUrl
 * @param {object} callbackObj callbackObj
 * @param {Function} callbackObj.resolve resolve
 * @param {Function} callbackObj.reject reject
 * @returns
 */
function unityLoader(src, { resolve, reject }) {
    if (!isBrowser)
        return;
    if (!src) {
        reject && reject(new Error(`${msgPrefix} loaderUrl not found.`));
        return;
    }
    /* if (typeof window.createUnityInstance === 'function') {
      log('Unity Loader already exists')
      resolve && resolve()
      return
    } */
    function handler(code) {
        if (code === 'ready') {
            resolve && resolve();
        }
        else {
            reject && reject(new Error(`${msgPrefix} ${src} loading failure.`));
        }
    }
    let script = window.document.querySelector(`script[src="${src}"]`);
    if (script === null) {
        script = window.document.createElement('script');
        script.type = 'text/javascript';
        script.src = src;
        script.async = true;
        script.setAttribute('data-status', 'loading');
        window.document.body.appendChild(script);
        const setAttributeFromEvent = function (event) {
            const _status = event.type === 'load' ? 'ready' : 'error';
            script === null || script === void 0 ? void 0 : script.setAttribute('data-status', _status);
            // handler(_status)
        };
        script.addEventListener('load', setAttributeFromEvent);
        script.addEventListener('error', setAttributeFromEvent);
    }
    else {
        handler(script.getAttribute('data-status'));
    }
    const setStateFromEvent = function (event) {
        handler(event.type === 'load' ? 'ready' : 'error');
    };
    script.addEventListener('load', setStateFromEvent);
    script.addEventListener('error', setStateFromEvent);
    return function () {
        if (script) {
            script.removeEventListener('load', setStateFromEvent);
            script.removeEventListener('error', setStateFromEvent);
            window.document.body.removeChild(script);
        }
    };
}

let BRIDGE_NAME = '__UnityLib__';
const defaultConfig = {
    streamingAssetsUrl: 'StreamingAssets',
    companyName: 'Unity',
    productName: 'Unity'
};
/**
 * generate UnityInstance arguments
 * @param {object} unityContext unityContext
 * @returns
 */
function generateUnityArguments(unity) {
    const unityInstanceArgs = Object.assign({}, unity.unityConfig);
    unityInstanceArgs.print = function (message) {
        unity.emit('debug', message);
    };
    unityInstanceArgs.printError = function (message) {
        unity.emit('error', message);
    };
    // delete unityInstanceArgs['lodaderUrl']
    return unityInstanceArgs;
}
/**
 * UnityWebgl
 * // new UnityWebgl(canvasElement, unityConfig, bridge)
 */
class UnityWebgl extends EventBus {
    constructor(canvas, config, bridge) {
        let window;
        window = globalThis || window;
        bridge = bridge !== null && bridge !== void 0 ? bridge : BRIDGE_NAME;
        if (isPlainObject(canvas) && typeof config === 'string') {
            bridge = config || bridge;
        }
        if (bridge in window) {
            log.error(msgPrefix + `window.${bridge} already exists.`);
        }
        BRIDGE_NAME = bridge;
        super((window[bridge] = {}));
        this.canvasElement = null;
        this.unityLoader = null;
        this.unityInstance = null;
        if (isPlainObject(canvas)) {
            this.unityConfig = Object.assign({}, defaultConfig, canvas);
        }
        else {
            this.unityConfig = Object.assign({}, defaultConfig, config);
            const $canvas = queryCanvas(canvas);
            if ($canvas) {
                this.create($canvas);
            }
        }
    }
    get bridge() {
        return BRIDGE_NAME;
    }
    set bridge(name) {
        //!window
        window[name] = window[BRIDGE_NAME];
        delete window[BRIDGE_NAME];
        BRIDGE_NAME = name;
    }
    /**
     * 创建 Unity应用实例并渲染至画布
     * @param canvas 画布
     * @returns
     */
    create(canvas) {
        if (!isBrowser)
            return;
        if (this.unityInstance && this.canvasElement && this.unityLoader) {
            log.warn('Unity Instance already exists!');
            return void 0;
        }
        const canvasElement = queryCanvas(canvas);
        if (!canvasElement) {
            log.warn('CanvasElement not found.');
            return void 0;
        }
        this.canvasElement = canvasElement;
        const ctx = this;
        const unityArguments = generateUnityArguments(this);
        this.emit('beforeMount', this);
        this.unityLoader = unityLoader(this.unityConfig.loaderUrl, {
            resolve() {
                try {
                    // Creates the Unity Instance, this method is made available globally by the Unity Loader.
                    window
                        .createUnityInstance(canvasElement, unityArguments, (val) => ctx.emit('progress', val))
                        .then((unity) => {
                        ctx.unityInstance = unity;
                        ctx.emit('created', unity); // todo 待删除
                        ctx.emit('mounted', ctx);
                    })
                        .catch((err) => {
                        throw err;
                    });
                }
                catch (err) {
                    ctx.unityInstance = null;
                    ctx.emit('error', err);
                }
            },
            reject(err) {
                log.error(err.message);
                ctx.emit('error', err);
            }
        });
    }
    /**
     * 销毁并重新加载Unity应用
     * @param config 配置项
     */
    reload(config) {
        var _a, _b;
        const canvasElement = this.canvasElement || ((_b = (_a = this.unityInstance) === null || _a === void 0 ? void 0 : _a.Module) === null || _b === void 0 ? void 0 : _b.canvas);
        if (this.unityInstance && canvasElement) {
            this.unityInstance.Quit().then(() => {
                this.unityLoader = null;
                this.unityConfig = Object.assign({}, this.unityConfig, config);
                this.emit('reload', this);
                this.create(canvasElement);
            }).catch(err => {
                this.emit('error', err);
            });
        }
    }
    /**
     * Sends a message to the UnityInstance to invoke a public method.
     * @param {string} objectName Unity scene name.
     * @param {string} methodName public method name.
     * @param {any} params an optional method parameter.
     * @returns
     */
    send(objectName, methodName, params) {
        if (this.unityInstance) {
            if (params === undefined || params === null) {
                this.unityInstance.SendMessage(objectName, methodName);
            }
            else {
                const _params = typeof params === 'object' ? JSON.stringify(params) : params;
                this.unityInstance.SendMessage(objectName, methodName, _params);
            }
        }
        else {
            log.warn('Unable to Send Message while Unity is not Instantiated.');
        }
        return this;
    }
    /**
     * Asynchronously ask for the pointer to be locked on current canvas. To track
     * the success or failure of the request, it is necessary to listen for the
     * pointerlockchange and pointerlockerror events at the Document level.
     * @public
     * @see https://developer.mozilla.org/en-US/docs/Web/API/Element/requestPointerLock
     */
    requestPointerLock() {
        var _a, _b;
        const canvasElement = this.canvasElement || ((_b = (_a = this.unityInstance) === null || _a === void 0 ? void 0 : _a.Module) === null || _b === void 0 ? void 0 : _b.canvas);
        if (canvasElement) {
            canvasElement.requestPointerLock();
        }
    }
    /**
     * Takes a screenshot of the canvas and returns a data URL containing image
     * data. The image data is in .png format unless otherwise specified.
     * @param {string} dataType The image format of the screenshot, ["image/png" | "image/jpeg" | "image/webp"]
     * @param {number} quality The quality of the jpg or webp screenshot
     * @returns a data URL containing image data of a snapshot of the canvas
     */
    takeScreenshot(dataType = 'image/jpeg', quality) {
        var _a, _b, _c;
        const canvasElement = this.canvasElement || ((_b = (_a = this.unityInstance) === null || _a === void 0 ? void 0 : _a.Module) === null || _b === void 0 ? void 0 : _b.canvas);
        if (!canvasElement) {
            log.warn('Unable to Take Screenshot while Unity is not Instantiated or Canvas is not available.');
            return;
        }
        if (((_c = this.unityConfig.webglContextAttributes) === null || _c === void 0 ? void 0 : _c.preserveDrawingBuffer) !== true) {
            log.warn('Taking a screenshot requires "preserveDrawingBuffer".');
        }
        // Takes a screenshot by converting Canvas's render-context's buffer into
        // a Data URL of the specified data type and quality.
        return canvasElement.toDataURL(dataType, quality);
    }
    /**
     * Enables or disabled the Fullscreen mode of the Unity Instance.
     * @param {boolean} enabled
     */
    setFullscreen(enabled) {
        if (!this.unityInstance) {
            // Guarding the Unity Instance.
            log.warn('Unable to Set Fullscreen while Unity is not Instantiated.');
            return;
        }
        this.unityInstance.SetFullscreen(enabled ? 1 : 0);
    }
    /**
     * Quits the Unity instance and clears it from memory so that Unmount from the DOM.
     */
    unload() {
        if (!this.unityInstance) {
            // Guarding the Unity Instance.
            log.warn('Unable to Quit Unity while Unity is not Instantiated.');
            return Promise.reject(new Error('Unity is not Instantiated.'));
        }
        this.emit('beforeUnmount', this);
        // Unmount unity.loader.js from the DOM
        if (typeof this.unityLoader === 'function') {
            this.unityLoader();
            this.unityLoader = null;
        }
        return this.unityInstance.Quit().then(() => {
            this.unityInstance = null;
            this.canvasElement = null;
            delete window[BRIDGE_NAME];
            this.emit('unmounted');
            this.emit('destroyed'); // todo 待删除
        }).catch((err) => {
            log.error('Unable to Unload Unity');
            this.emit('error', err);
            throw err;
        });
    }
    /**
     * Detatches the Unity Instance from the React DOM, by doing so, the Unity
     * Instance can be unloaded from the memory while the Unity component can be
     * unmounted safely.
     *
     * Warning! This is a workaround for the fact that the Unity WebGL instances
     * which are build with Unity 2021.2 and newer cannot be unmounted before the
     * Unity Instance is unloaded.
     */
    _unsafe_unload() {
        try {
            if (this.unityInstance === null || !this.unityInstance.Module.canvas) {
                // Guarding the Unity Instance.
                log.warn('No Unity Instance found.');
                return Promise.reject();
            }
            // Re-attaches the canvas to the body element of the document. This way it
            // wont be removed from the DOM when the component is unmounted. Then the
            // canvas will be hidden while it is being unloaded.
            const canvas = this.unityInstance.Module.canvas;
            document.body.appendChild(canvas);
            canvas.style.display = 'none';
            // Unloads the Unity Instance.
            return this.unload().then(() => {
                // Eventually the canvas will be removed from the DOM. This has to be done
                // manually since the canvas is no longer controlled by the React DOM.
                canvas.remove();
            });
        }
        catch (e) {
            log.error(e.message);
            return Promise.reject(e);
        }
    }
}

module.exports = UnityWebgl;
